/*****************************************************************************\
*                                                                             *
*   FP2CCD uses the data provided in a CAMERA DEFINITION FILE and a           *
*          CAMERA OFFSET FILE to transform                                    *
*          coordinates from the spectrograph focal plane to pixel locations   *
*          on the ccd camera. Returns chip # and on-chip coordinates          *
*                                                                             *
*   READCDF reads in the camera definition file data and must be called       *
*           before the first call to fp2ccd                                   *
*                                                                             *
*   READCOF reads in the dewar offset  file data and must be called           *
*           before the first call to fp2ccd                                   *
*                                                                             *
*   SETUPCAMERA inputs obsdef info for possible using in mapping corrections  *
*               and calls READCDF and READCOF
*                                                                             *
*   GETCHIPDAT returns chip data (number,size)                                *
*                                                                             *
*   CCD8 puts chip coordinates on an 8-chip scale                             *
*                                                                             *
*   MOSPOS puts chip coordinates on mosaic scale                              *
*                                                                             *
*   VERSION 02 May 06                                                         *
*                                                                             *
\*****************************************************************************/

#include  <stdlib.h>
#include  <string.h>
#include  <stdio.h>
#include   <math.h>
#include  "cosmos.h"

static int    nchip,xchip,ychip,xz[8],sxz[8],yz[8],syz[8],sl_n,sts1_n,stt1_n,
              i,oldsite,sts2_n,stt2_n,srs1_n,srt1_n,srs2_n,srt2_n,srz_n,xarsize,
              yarsize,hand,xxn[8],yyn[8],distor,oldway;
static float  scale,xmin[8],xmax[8],ymin[8],ymax[8],dtheta[8],xx0[8],yy0[8],
              srscl0,srscl1,srsclth,dcth,dcx,dcy,dscale,sts1_coef[4],sl_coef[4],
              dt,stt1_coef[4],sts2_coef[4],stt2_coef[4],srs1_coef[4],
              drr,srt1_coef[4],srs2_coef[4],srt2_coef[4],srz_coef[4];
static char   grating[20],distfunc[80];
static obsdef obsinfo;

//
// SetupCamera   calls Readcdf and Readcof and inputs instrument setup infor
// for possible use in correcting mapping
//

int SetupCamera(obsdef obsdat){
  
  obsinfo=obsdat;
  if(Readcdf(obsdat.dewar)==1){
    printf("Error reading dewar definition file %s\n",obsdat.dewar);
    return 1;}
  if(Readcof(obsdat.dewoff)==1){
    printf("Error reading dewar offset file %s\n",obsdat.dewoff);
    return 1;}
  return 0;}

//
// Readcdf
//

int Readcdf(char filename[]){

  FILE     *parmfile;
  char     CHAR,line[133],imacsdir[130];
  int      i,nopar;
  char     *COS_HOME;

  COS_HOME=malloc(sizeof(CHAR)*80);
  COS_HOME=getenv("COSMOS_HOME");
  if(COS_HOME==NULL){
    printf("COSMOS_HOME undefined!\n");
    return 1;}
  strcpy(imacsdir,COS_HOME);
  strcat(imacsdir,"/sdata/");
  strcat(imacsdir,filename);
  strcat(imacsdir,".dewdef");
  dscale=1.;
  if((parmfile=fopen(imacsdir,"r"))==NULL){
    printf("Cannot find dewar definition file\n");
    return 1;}
  for(;;){
    if(!fgets(line,133,parmfile)){
      fclose(parmfile);
      return 1;}
    if(*line=='#') continue;
    i=sscanf(line,"%d %d %d %f %d %d %d",&nchip,&xchip,&ychip,&scale,&xarsize,
             &yarsize,&hand);
    //handling legacy IMACS SITE.dewdef files
    if(i==4){
      printf("\nWarning: you are using the old SITE.dewdef file\n\n");
      xarsize=yarsize=8192;
      hand=0;
      oldsite=1;}

    xchip-=1;
    ychip-=1;
    break;}
  for(i=0;i<nchip;i++){
    for(;;){
      if(!fgets(line,133,parmfile)){
	fclose(parmfile);
	return 1;}
      if(*line=='#') continue;
      sscanf(line,"%f %f %f %f %d %d",&xmin[i],&xmax[i],&ymin[i],&ymax[i],
             &xxn[i],&yyn[i]);
      break;}
    for(;;){
      if(!fgets(line,133,parmfile)){
	fclose(parmfile);
	return 1;}
      if(*line=='#') continue;
      sscanf(line,"%d %d %d %d",&xz[i],&sxz[i],&yz[i],&syz[i]);
      break;}
    for(;;){
      if(!fgets(line,133,parmfile)){
	fclose(parmfile);
	return 1;}
      if(*line=='#') continue;
      sscanf(line,"%f",&dtheta[i]);
      dtheta[i]/=57.2958;
      break;}
    for(;;){
      if(!fgets(line,133,parmfile)){
	fclose(parmfile);
	return 1;}
      if(*line=='#') continue;
      sscanf(line,"%f %f",&xx0[i],&yy0[i]);                     
      break;}
    }

  if(oldsite){
    yyn[0]=yyn[1]=yyn[2]=yyn[3]=2;
    yyn[4]=yyn[5]=yyn[6]=yyn[7]=0;
    xxn[0]=0;
    xxn[1]=1;
    xxn[2]=2;
    xxn[3]=3;
    xxn[4]=2;
    xxn[5]=1;
    xxn[6]=4;
    xxn[7]=3;}

  //Short Camera Distortion corrections

  distor=oldway=0;
  sl_n=0;
  for(;;){
    if(!fgets(line,133,parmfile)) break;
    if(!strcmp(line,"none\n")) break;
    //Old way of defining distortions
    if(strstr(line,"Short camera position corrections")!=NULL){
      oldway=1;
      nopar=1;
      while(1){
        fgets(line,133,parmfile);
        if(*line=='#') continue;
        if(!strcmp(line,"none\n")) break;
        nopar=0;
        break;}
      if(nopar){
          //no distortion parameters, must be old E2V file, fixup
        printf("fixing up old E2V.dewdef file\n");
        oldway=0;
        distor=1;
        sl_n=3;sl_coef[0]=-5.30366; sl_coef[1]=2.478044e-3; 
        sl_coef[2]=-3.691942e-7;  sl_coef[3]=1.720668e-11;
        strcpy(distfunc,"E2V1");
        fclose(parmfile);
        return 0;}
      sscanf(line,"%d %e %e %e %e",&sl_n,&sl_coef[0],&sl_coef[1],
             &sl_coef[2],&sl_coef[3]);
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
        return 0;}
      distor=1;
      sscanf(line,"%d %e %e %e %e",&sts1_n,&sts1_coef[0],&sts1_coef[1],
             &sts1_coef[2],&sts1_coef[3]);
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
        return 0;}
      sscanf(line,"%d %e %e %e %e",&stt1_n,&stt1_coef[0],&stt1_coef[1],
             &stt1_coef[2],&stt1_coef[3]);
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
        return 0;}
      sscanf(line,"%d %e %e %e %e",&sts2_n,&sts2_coef[0],&sts2_coef[1],
             &sts2_coef[2],&sts2_coef[3]);
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
        return 0;}
      sscanf(line,"%d %e %e %e %e %e",&stt2_n,&stt2_coef[0],&stt2_coef[1],
             &stt2_coef[2],&stt2_coef[3]);
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
        return 0;}
      sscanf(line,"%d %e %e %e %e",&srs1_n,&srs1_coef[0],&srs1_coef[1],
             &srs1_coef[2],&srs1_coef[3]);
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
        return 0;}
      sscanf(line,"%d %e %e %e %e",&srt1_n,&srt1_coef[0],&srt1_coef[1],
             &srt1_coef[2],&srt1_coef[3]);
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
        return 0;}
      sscanf(line,"%d %e %e %e %e",&srs2_n,&srs2_coef[0],&srs2_coef[1],
             &srs2_coef[2],&srs2_coef[3]);
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
        return 0;}
      sscanf(line,"%d %e %e %e %e %e",&srt2_n,&srt2_coef[0],&srt2_coef[1],
             &srt2_coef[2],&srt2_coef[3]);
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
      return 0;}
      sscanf(line,"%d %e %e %e %e %e",&srz_n,&srz_coef[0],&srz_coef[1],
             &srz_coef[2],&srz_coef[3]);
      break;}
    //new way of handling distortions
    else if(strstr(line,"LAMBDA CORRECTION")!=NULL){
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
        return 0;}
      sscanf(line,"%d %e %e %e %e",&sl_n,&sl_coef[0],&sl_coef[1],
             &sl_coef[2],&sl_coef[3]);}
    else if(strstr(line,"DISTORTION CORRECTION")!=NULL){
      distor=1;
      if(!fgets(line,133,parmfile)){
        fclose(parmfile);
        return 0;}
      sscanf(line,"%s",distfunc);
      printf("Using %s distortion corrections\n",distfunc);
      break;}
  }
  fclose(parmfile);
  return 0;}

    

//
//Readcof
//

int Readcof(char filename[]){

  FILE     *parmfile;
  char     line[133];
  int      i;

  strcpy(line,filename);
  if(strstr(line,".dewoff")==NULL) strcat(line,".dewoff");
 if((parmfile=fopen(line,"r"))==NULL){
    return 1;}
  for(;;){
    if(!fgets(line,133,parmfile)){
      fclose(parmfile);
      return 1;}
    if(*line=='#') continue;
    sscanf(line,"%f %f %f %f",&dcth,&dscale,&dcx,&dcy);
    dcth/=57.2958;
    break;}
  fclose(parmfile);
  return 0;}

//
//Getchipdat
//

void Getchipdat(dewdat* dewinfo){

  dewinfo->nchip=nchip; //#chips
  dewinfo->xchip=xchip+1; //#x pixels
  dewinfo->ychip=ychip+1; //#y pixels
  dewinfo->xarsize=xarsize; //array size
  dewinfo->yarsize=yarsize;
  dewinfo->hand=hand; //chips contiguous?
  dewinfo->scale=scale/dscale;
  dewinfo->theta=dcth;
  for(i=1;i<=nchip;i++){
    dewinfo->sx[i]=sxz[i-1];
    dewinfo->sy[i]=syz[i-1];
    dewinfo->xz[i]=xz[i-1];
    dewinfo->yz[i]=yz[i-1];
    dewinfo->xloc[i]=xxn[i-1];
    dewinfo->yloc[i]=yyn[i-1];}
  return;}

//
//fp2ccd
//

int fp2ccd(float xfp, float yfp, float *xccd, float *yccd, float lambda){
  
  int       i,chip;
  double     r,theta,cth;
  float     dscl,xx;
  //moe parameters
  float     moecoef[6]={1.83521176e-01,-7.49109947e-04,-1.21785697e-04,
  -8.07591646e-07,1.38731509e-08,1.67187628e-10};
  
  /*-------------------correct for dewar offsets-----------------------------*/

  r=sqrt((double)xfp*(double)xfp+(double)yfp*(double)yfp); 
  if(xfp==0){
    if(yfp==0){
      theta=0;}
    else{
      theta=1.5707963;
      if(yfp<0){
        theta=-theta;}
      }
    }
  else if(yfp==0){
    theta=0;
    if(xfp<0){
      theta=3.141592654;}
    }
  else{
    theta= atan(yfp/xfp);
    if(xfp<0){
      if(yfp>0){
        theta+=3.141592654;}
      else{
        theta-=3.141592654;}
      }
    }
  dscl=dscale;
  
  //IMACS scale,distortion corrections
  if(strstr(obsinfo.instrument,"IMACS") != NULL){
    //MOE
    if(strstr(obsinfo.grating,"MOE") != NULL){
      xx=polyvalue(xfp,moecoef,5);
      xfp-=xx-.07*(obsinfo.gr_order-6);
      r=sqrt((double)xfp*(double)xfp+(double)yfp*(double)yfp);
      cth=xfp/r;
      if(cth>-1. && cth<1.){
        theta= acos(cth);}
      else{
        theta = (cth>0) ? 0.: 1.570796;}
      if(yfp<0) theta=-theta;
      }
    //short camera
    if(strcmp(obsinfo.camera,"SHORT") ==0){
      //scale vs lambda
      if(sl_n){
        dscl=dscale*(1.0 + 0.001*polyvalue(lambda,sl_coef,sl_n));}
      //distortion map
      if(distor){
        if(oldway){
            //d theta
          dt= 0.001*(polyvalue(r,sts1_coef,sts1_n)*cos((double)(theta-polyvalue(
              r,stt1_coef,stt1_n)))+polyvalue(r,sts2_coef,sts2_n)*cos(2*(double)
              (theta-polyvalue(r,stt2_coef,stt2_n))));
          theta-=dt;
      //d r
          drr=0.001*(polyvalue(r,srs1_coef,srs1_n)*cos((double)(theta-polyvalue(
              r,srt1_coef,srt1_n)))+polyvalue(r,srs2_coef,srs2_n)*cos(2*(double)
              (theta-polyvalue(r,srt2_coef,srt2_n)))+polyvalue(r,srz_coef,
              srz_n));
          r*=(1.0-drr);}
        else if(strstr(distfunc,"E2V1")!=NULL){
          e2v1(&theta,&r);}
        else if(strstr(distfunc,"E2V2")!=NULL){
          e2v2(&theta,&r);}
      }
    }
    //long camera
  }

  theta-=dcth;
  xfp=r*cos(theta);
  yfp=r*sin(theta);
  xfp/=dscl;
  yfp/=dscl;
  xfp-=dcx;
  yfp-=dcy;

  // one contiguous chip

  if(nchip==1){
    yfp*=hand;                         //parity vs imacs dewar
    *xccd=xfp*scale+xarsize/2;
    *yccd=yfp*scale+yarsize/2;
    if(*xccd>=0 && *xccd<xarsize && *yccd>=0 && *yccd<yarsize){
      chip=1;}
    else{
      chip=-1;}
    return chip;}

  //multiple chips

  else{

    //identify chip

    chip=-1;
    for(i=0;i<nchip;i++){
      if(xfp<xmin[i] || xfp>xmax[i] || yfp<ymin[i] || yfp>ymax[i]) continue;
      chip=i;
      break;}
    if(chip<0){
      chip=0;
      return chip;}

    //transform coordinates

    xfp-=xx0[chip];
    yfp-=yy0[chip];
    r=sqrt(xfp*xfp+yfp*yfp);
    theta=acos((double)xfp/r);
    if(yfp<0) theta=-theta;
    theta-=dtheta[chip];
    r*=scale;
    xfp=r*cos(theta);
    yfp=r*sin(theta);
    *xccd=xz[chip]+sxz[chip]*xfp;
    *yccd=yz[chip]+syz[chip]*yfp;
    if(*xccd>=0 && *xccd<=xchip && *yccd>=0 & *yccd<=ychip){
      chip+=1;}
    else{
      chip=-1-chip;}
    return chip;}
}

int ccd8(int chip, float xccd, float yccd, float *x8, float *y8){

  int chp;
 
  if(nchip==1){
    *x8=xccd;
    *y8=hand*(yccd-yarsize/2);}
  else{
    chp=chip-1;
    *y8=yy0[chp]*scale +yz[chp]+syz[chp]*yccd;
    *x8=xx0[chp]*scale +xz[chp]+sxz[chp]*xccd;}
  return 0;}

					  
/*
 *
 *  mspos  converts ccd coordinates to mosaic coordinates
 *
 */

int mspos(int chip,float xccd,float yccd,float *x8cd,float *y8cd){

  int chop;
  
  if(chip<0) chip=-chip;
  chop=chip-1;
   if(nchip==1){
    *x8cd=xccd;
    *y8cd=yccd;
    if(sxz[chop]<0) *x8cd--;
    if(syz[chop]<0) *y8cd--;
   return 0;}
  *x8cd=xxn[chop]*(xchip+1)+sxz[chop]*xccd;
  if(sxz[chop]<0) *x8cd=*x8cd-1;
  *y8cd=yyn[chop]*(ychip+1)+syz[chop]*yccd;
  if(syz[chop]<0) *y8cd=*y8cd-1;
  return 0;}

/*
 *  DISTORTION MAPS
 */

/*
 *
 *  distort2: SHORT CA<ERA xx 2008 - yy 2009
 *
 */
 
int e2v1( double *thet, double *rr){  
  double rad, th, dr, dth, cth;
  
  rad=*rr;
  th=*thet;
  cth=cos((*thet-2.25)/2.);
  dr=-0.00370153-0.142815*cth-0.422365*pow(cth,2.)-0.250696*pow(cth,3.) 
      +0.0851477*pow(cth,4.)+0.45882*pow(cth,5.) +0.842774*pow(cth,6.);
  dr*=0.167*(-0.138853 +0.0212971*rad -4.47067e-4*pow(rad,2.) +4.67963e-6*pow(rad,3.));
  dth= -7.17805e-5 +0.000144754*th +0.000205818*pow(th,2.) -5.79749e-5*pow(th,3.)
    -6.32755e-5*pow(th,4.) +4.68534e-6*pow(th,5.) +3.9343e-6*pow(th,6.);
//  printf("%10.5f %10.5f %10.5f %10.5f\n",th, rad, cth, dr);
  *thet-=dth;
  *rr-=dr;
  return 0;}

int e2v2(float *thet, float *rr){
  
  return 0;}
            
  
  